#ifndef LEDA_INTEGER_VECTOR_H
#define LEDA_INTEGER_VECTOR_H

//------------------------------------------------------------------------------
//  vectors
//------------------------------------------------------------------------------

#include "numberdef"
#include <LEDA/basic.h>
#include <LEDA/integer.h>



/*{\Manpage {integer_vector} {} {Integer-Valued Vectors}}*/

class integer_vector
{

/*{\Mdefinition
An instance of the data type $integer\_vector$ is a vector of real variables.}*/

  friend class integer_matrix;

  number* v;
  int d;

  void check_dimensions(const integer_vector&) const;
 
public:

/*{\Mcreation v }*/

integer_vector(); 
/*{\Mcreate creates an instance $v$ of type $vector$; $v$ is initialized to 
            the zero-dimensional vector.}*/

integer_vector(int d); 
/*{\Mcreate creates an instance $v$ of type $vector$; $v$ is initialized to 
            the zero vector of dimension $d$.}*/ 

integer_vector(number a, number b);
/*{\Mcreate creates an instance $v$ of type $vector$; $v$ is initialized to 
            the two-dimensional vector $(a,b)$.}*/

integer_vector(number a, number b, number c);
/*{\Mcreate creates an instance $v$ of type $vector$; $v$ is initialized to 
            the three-dimensional vector $(a,b,c)$.}*/

  integer_vector(const integer_vector&);
 ~integer_vector(); 
  integer_vector& operator=(const integer_vector&);


/*{\Moperations 2 4.5}*/

int    dim()    const { return d; }
/*{\Mop       returns the dimension of $v$.}*/ 

//double length() const;
/*{\Mop      returns the Euclidean length of $v$.}*/
  
//integer_vector norm()   const { return *this/length(); }
  
//double angle(const integer_vector& w) const; 
/*{\Mop     returns the angle between $v$ and $w$.}*/

  
number& operator[](int i)
/*{\Marrop     returns $i$-th component of $v$.\\
	       \precond $0\le i \le v$.dim()$-$1.}*/
{ 
  #ifndef MY_OFF
  if (i<0 || i>=d)  error_handler(1,"vector: index out of range ");
  #endif
  return v[i]; 
}
  
number  operator[](int i) const
{ 
  #ifndef MY_OFF
  if (i<0 || i>=d)  error_handler(1,"vector: index out of range ");
  #endif
  return v[i]; 
}


integer_vector& operator+=(const integer_vector&);
integer_vector& operator-=(const integer_vector&);
  
integer_vector  operator+(const integer_vector& v1) const;
/*{\Mbinop     Addition.\\
	       \precond $v$.dim() = $v1$.dim().}*/

integer_vector  operator-(const integer_vector& v1) const;
/*{\Mbinop     Subtraction.\\
	       \precond $v$.dim() = $v1$.dim().}*/

number  operator*(const integer_vector& v1) const;
/*{\Mbinop     Scalar multiplication.\\
               \precond $v$.dim() = $v1$.dim().}*/

integer_vector  operator*(number r)        const;
/*{\Mbinop     Componentwise multiplication with number $r$.}*/

integer_vector  operator-() const;
//vector  operator/(double)        const;
  
bool     operator==(const integer_vector& w) const;
/*{\Mbinop     Test for equality.}*/

bool     operator!=(const integer_vector& w)  const { return !(*this == w); }
/*{\Mbinop     Test for inequality.}*/

  

friend integer_vector operator*(number f, const integer_vector& v);
/*
friend integer_vector operator/(const vector& v, number f);
*/
  
friend ostream& operator<<(ostream& O, const integer_vector& v);
/*{\Mbinopfunc  writes $v$ componentwise to the output stream $O$.}*/

friend istream& operator>>(istream& I, integer_vector& v);
/*{\Mbinopfunc  reads $v$ componentwise from the input stream $I$.}*/

static int cmp(const integer_vector&, const integer_vector&);
 
LEDA_MEMORY(integer_vector)

};

inline int compare(const integer_vector& x, const integer_vector& y)
{ return integer_vector::cmp(x,y); }


inline void Print(const integer_vector& v, ostream& out) { out << v; }
inline void Read(integer_vector& v, istream& in)         { in >> v;  }

LEDA_TYPE_PARAMETER(integer_vector)

/*{\Mimplementation
Vectors are implemented by arrays of integers. All operations on a vector 
$v$ take time $O(v.dim())$, except for dim and $[\ ]$ which take constant
time. The space requirement is $O(v.dim())$.
}*/


#endif
